// Product Data - Based on the actual images in the APPAREL folder
const products = [
    {
        id: 1,
        name: "Unishop Bermuda Shorts",
        price: 29.99,
        image: "APPAREL/Bermuda Shorts PNG.png",
        category: "Shorts",
        sizes: ["S", "M", "L", "XL"],
        description: "Comfortable and stylish Bermuda shorts perfect for casual wear."
    },
    {
        id: 2,
        name: "Unishop Black Hoodie",
        price: 49.99,
        image: "APPAREL/Black Hoodie.png",
        category: "Hoodie",
        sizes: ["S", "M", "L", "XL"],
        description: "Premium quality black hoodie with soft fabric and comfortable fit."
    },
    {
        id: 3,
        name: "Unishop Black Leather Zip-up Jacket",
        price: 89.99,
        image: "APPAREL/Black Leather Zip-up Jacket.png",
        category: "Jacket",
        sizes: ["S", "M", "L", "XL"],
        description: "Stylish black leather jacket with zip-up design for ultimate style."
    },
    {
        id: 4,
        name: "Unishop Blank Pants",
        price: 34.99,
        image: "APPAREL/Blank Pants.png",
        category: "Pants",
        sizes: ["S", "M", "L", "XL"],
        description: "Versatile blank pants suitable for various occasions."
    },
    {
        id: 5,
        name: "Unishop Bomber Jacket",
        price: 79.99,
        image: "APPAREL/Bomber Jacket.png",
        category: "Jacket",
        sizes: ["S", "M", "L", "XL"],
        description: "Classic bomber jacket with modern styling and premium materials."
    },
    {
        id: 6,
        name: "Unishop Cargo Pants Blue",
        price: 39.99,
        image: "APPAREL/Cargo Pants Blue.png",
        category: "Pants",
        sizes: ["S", "M", "L", "XL"],
        description: "Functional blue cargo pants with multiple pockets for utility."
    },
    {
        id: 7,
        name: "Unishop Cargo Pants Green",
        price: 39.99,
        image: "APPAREL/Cargo Pants Green.png",
        category: "Pants",
        sizes: ["S", "M", "L", "XL"],
        description: "Durable green cargo pants perfect for outdoor activities."
    },
    {
        id: 8,
        name: "Unishop Cargo Pants",
        price: 39.99,
        image: "APPAREL/Cargo Pants.png",
        category: "Pants",
        sizes: ["S", "M", "L", "XL"],
        description: "Essential cargo pants with practical design and comfort."
    },
    {
        id: 9,
        name: "Unishop Colorful Hoodie",
        price: 52.99,
        image: "APPAREL/Colorful Hoodie.png",
        category: "Hoodie",
        sizes: ["S", "M", "L", "XL"],
        description: "Vibrant colorful hoodie that adds personality to your wardrobe."
    },
    {
        id: 10,
        name: "Unishop Green Jacket",
        price: 69.99,
        image: "APPAREL/Green Jacket PNG.png",
        category: "Jacket",
        sizes: ["S", "M", "L", "XL"],
        description: "Stylish green jacket with modern cut and premium finish."
    },
    {
        id: 11,
        name: "Unishop Green Polo Shirt",
        price: 24.99,
        image: "APPAREL/Green Polo Shirt PNG.png",
        category: "Shirt",
        sizes: ["S", "M", "L", "XL"],
        description: "Classic green polo shirt perfect for casual and semi-formal occasions."
    },
    {
        id: 12,
        name: "Unishop Grey Sweatshirt",
        price: 44.99,
        image: "APPAREL/Grey Sweatshirt.png",
        category: "Sweatshirt",
        sizes: ["S", "M", "L", "XL"],
        description: "Comfortable grey sweatshirt for relaxed everyday wear."
    },
    {
        id: 13,
        name: "Unishop Hiking Pants",
        price: 54.99,
        image: "APPAREL/Hiking Pants.png",
        category: "Pants",
        sizes: ["S", "M", "L", "XL"],
        description: "Durable hiking pants designed for outdoor adventures and comfort."
    },
    {
        id: 14,
        name: "Unishop Khaki Shorts",
        price: 27.99,
        image: "APPAREL/Khaki Shorts.png",
        category: "Shorts",
        sizes: ["S", "M", "L", "XL"],
        description: "Classic khaki shorts perfect for summer and casual outings."
    },
    {
        id: 15,
        name: "Unishop Red T-Shirt",
        price: 19.99,
        image: "APPAREL/Red T-Shirt PNG.png",
        category: "T-Shirt",
        sizes: ["S", "M", "L", "XL"],
        description: "Essential red t-shirt made from premium cotton for ultimate comfort."
    }
];

// Global variables
let cart = JSON.parse(localStorage.getItem('cart')) || [];
let wishlist = JSON.parse(localStorage.getItem('wishlist')) || [];
let currentSlide = 0;

// Clear any old localStorage data and reset
function clearOldData() {
    // Only clear if there's corrupted data
    if (cart.some(item => !item.id || !item.name)) {
        localStorage.removeItem('cart');
        localStorage.removeItem('wishlist');
        cart = [];
        wishlist = [];
    }
    updateCartCount();
    updateWishlistCount();
}

// Initialize data on page load
clearOldData();

// Initialize the website
document.addEventListener('DOMContentLoaded', function () {
    initializeHeroCarousel();
    loadFeaturedProducts();
    loadBestsellers();
    updateCartCount();
    updateWishlistCount();
    initializeEventListeners();
    initializeNewsletter();
});

// Hero Carousel Functions
function initializeHeroCarousel() {
    const slides = document.querySelectorAll('.hero-slide');
    const dots = document.querySelectorAll('.dot');

    if (slides.length === 0) return;

    function showSlide(index) {
        slides.forEach((slide, i) => {
            slide.classList.toggle('active', i === index);
        });

        dots.forEach((dot, i) => {
            dot.classList.toggle('active', i === index);
        });

        currentSlide = index;
    }

    function nextSlide() {
        const nextIndex = (currentSlide + 1) % slides.length;
        showSlide(nextIndex);
    }

    // Auto-advance carousel every 5 seconds
    setInterval(nextSlide, 5000);

    // Add click handlers for dots
    dots.forEach((dot, index) => {
        dot.addEventListener('click', () => showSlide(index));
    });
}

// Make currentSlide function global for onclick handlers
window.currentSlide = function (index) {
    const slides = document.querySelectorAll('.hero-slide');
    const dots = document.querySelectorAll('.dot');

    slides.forEach((slide, i) => {
        slide.classList.toggle('active', i === index - 1);
    });

    dots.forEach((dot, i) => {
        dot.classList.toggle('active', i === index - 1);
    });

    currentSlide = index - 1;
};

// Product Functions
function loadFeaturedProducts() {
    const container = document.getElementById('featured-products');
    if (!container) return;

    // Show first 6 products as featured
    const featuredProducts = products.slice(0, 6);

    container.innerHTML = featuredProducts.map(product => `
        <div class="product-card" onclick="openProductModal(${product.id})">
            <div class="product-image">
                <img src="${product.image}" alt="${product.name}" loading="lazy">
            </div>
            <div class="product-info">
                <h3 class="product-title">${product.name}</h3>
                <p class="product-price">$${product.price}</p>
                <div class="product-actions">
                    <button class="btn btn-primary btn-small" onclick="event.stopPropagation(); addToCart(${product.id})">
                        Add to Cart
                    </button>
                    <button class="btn btn-secondary btn-small" onclick="event.stopPropagation(); toggleWishlist(${product.id})">
                        <i class="fas fa-heart"></i>
                    </button>
                </div>
            </div>
        </div>
    `).join('');
}

function loadBestsellers() {
    const container = document.querySelector('.bestsellers-track');
    if (!container) return;

    // Show bestsellers (mix of different categories)
    const bestsellers = [
        products[1], // Black Hoodie
        products[4], // Bomber Jacket
        products[14], // Red T-Shirt
        products[6], // Cargo Pants Blue
        products[9], // Colorful Hoodie
        products[12], // Grey Sweatshirt
        products[0], // Bermuda Shorts
        products[11] // Green Polo Shirt
    ];

    // Duplicate the array to create seamless loop
    const duplicatedBestsellers = [...bestsellers, ...bestsellers];

    container.innerHTML = duplicatedBestsellers.map(product => `
        <div class="bestseller-item">
            <img src="${product.image}" alt="${product.name}" loading="lazy">
            <h4>${product.name}</h4>
            <p>$${product.price}</p>
        </div>
    `).join('');
}

// Cart Functions
function addToCart(productId) {
    const product = products.find(p => p.id === productId);
    if (!product) return;

    const existingItem = cart.find(item => item.id === productId && item.size === 'M');

    if (existingItem) {
        existingItem.quantity += 1;
    } else {
        cart.push({
            id: productId,
            size: 'M',
            quantity: 1,
            ...product
        });
    }

    saveCart();
    updateCartCount();
    showNotification('Product added to cart!');
}

function removeFromCart(productId, size) {
    cart = cart.filter(item => !(item.id === productId && item.size === size));
    saveCart();
    updateCartCount();
    renderCartItems();
}

function updateCartQuantity(productId, size, newQuantity) {
    const item = cart.find(item => item.id === productId && item.size === size);
    if (item) {
        if (newQuantity <= 0) {
            removeFromCart(productId, size);
        } else {
            item.quantity = newQuantity;
            saveCart();
            renderCartItems();
        }
    }
}

function saveCart() {
    localStorage.setItem('cart', JSON.stringify(cart));
}

function updateCartCount() {
    const count = cart.reduce((total, item) => total + item.quantity, 0);
    const countElement = document.getElementById('cart-count');
    if (countElement) {
        countElement.textContent = count;
        countElement.style.display = count > 0 ? 'flex' : 'none';
    }
}

function renderCartItems() {
    const container = document.getElementById('cart-items');
    if (!container) return;

    if (cart.length === 0) {
        container.innerHTML = '<p style="text-align: center; color: #6b7280; padding: 2rem;">Your cart is empty</p>';
        return;
    }

    container.innerHTML = cart.map(item => `
        <div class="cart-item">
            <div class="cart-item-image">
                <img src="${item.image}" alt="${item.name}">
            </div>
            <div class="cart-item-info">
                <div class="cart-item-title">${item.name}</div>
                <div class="cart-item-price">$${item.price}</div>
                <div class="cart-item-size">Size: ${item.size}</div>
                <div class="quantity-controls">
                    <button class="quantity-btn" onclick="updateCartQuantity(${item.id}, '${item.size}', ${item.quantity - 1})">-</button>
                    <input type="number" class="quantity-input" value="${item.quantity}" min="1" onchange="updateCartQuantity(${item.id}, '${item.size}', parseInt(this.value))">
                    <button class="quantity-btn" onclick="updateCartQuantity(${item.id}, '${item.size}', ${item.quantity + 1})">+</button>
                </div>
                <button class="remove-btn" onclick="removeFromCart(${item.id}, '${item.size}')">Remove</button>
            </div>
        </div>
    `).join('');

    // Update total
    const total = cart.reduce((sum, item) => sum + (item.price * item.quantity), 0);
    const totalElement = document.getElementById('cart-total');
    if (totalElement) {
        totalElement.textContent = total.toFixed(2);
    }
}

// Wishlist Functions
function toggleWishlist(productId) {
    const product = products.find(p => p.id === productId);
    if (!product) return;

    const existingIndex = wishlist.findIndex(item => item.id === productId);

    if (existingIndex >= 0) {
        wishlist.splice(existingIndex, 1);
        showNotification('Product removed from wishlist!');
    } else {
        wishlist.push(product);
        showNotification('Product added to wishlist!');
    }

    saveWishlist();
    updateWishlistCount();
    renderWishlistItems();
}

function removeFromWishlist(productId) {
    wishlist = wishlist.filter(item => item.id !== productId);
    saveWishlist();
    updateWishlistCount();
    renderWishlistItems();
}

function saveWishlist() {
    localStorage.setItem('wishlist', JSON.stringify(wishlist));
}

function updateWishlistCount() {
    const count = wishlist.length;
    const countElement = document.getElementById('wishlist-count');
    if (countElement) {
        countElement.textContent = count;
        countElement.style.display = count > 0 ? 'flex' : 'none';
    }
}

function renderWishlistItems() {
    const container = document.getElementById('wishlist-items');
    if (!container) return;

    if (wishlist.length === 0) {
        container.innerHTML = '<p style="text-align: center; color: #6b7280; padding: 2rem;">Your wishlist is empty</p>';
        return;
    }

    container.innerHTML = wishlist.map(item => `
        <div class="wishlist-item">
            <div class="wishlist-item-image">
                <img src="${item.image}" alt="${item.name}">
            </div>
            <div class="wishlist-item-info">
                <div class="wishlist-item-title">${item.name}</div>
                <div class="wishlist-item-price">$${item.price}</div>
                <div class="product-actions">
                    <button class="btn btn-primary btn-small" onclick="addToCart(${item.id})">
                        Add to Cart
                    </button>
                    <button class="remove-btn" onclick="removeFromWishlist(${item.id})">Remove</button>
                </div>
            </div>
        </div>
    `).join('');
}

// Product Modal Functions
function openProductModal(productId) {
    const product = products.find(p => p.id === productId);
    if (!product) return;

    const modal = document.getElementById('product-modal');
    const modalBody = document.getElementById('modal-body');

    modalBody.innerHTML = `
        <div class="modal-product">
            <div class="modal-image">
                <img src="${product.image}" alt="${product.name}">
            </div>
            <div class="modal-info">
                <h2>${product.name}</h2>
                <div class="price">$${product.price}</div>
                <div class="description">${product.description}</div>
                <div class="size-selector">
                    <h4>Select Size:</h4>
                    <div class="size-options">
                        ${product.sizes.map(size => `
                            <button class="size-option" onclick="selectSize(this)" data-size="${size}">${size}</button>
                        `).join('')}
                    </div>
                </div>
                <div class="product-actions">
                    <button class="btn btn-primary btn-full" onclick="addToCartFromModal(${product.id})">
                        Add to Cart
                    </button>
                    <button class="btn btn-secondary btn-full" onclick="toggleWishlistFromModal(${product.id})">
                        <i class="fas fa-heart"></i> Add to Wishlist
                    </button>
                </div>
            </div>
        </div>
    `;

    modal.style.display = 'block';
}

function selectSize(button) {
    document.querySelectorAll('.size-option').forEach(btn => btn.classList.remove('selected'));
    button.classList.add('selected');
}

function addToCartFromModal(productId) {
    const selectedSize = document.querySelector('.size-option.selected');
    if (!selectedSize) {
        showNotification('Please select a size', 'error');
        return;
    }

    const product = products.find(p => p.id === productId);
    if (!product) return;

    const size = selectedSize.dataset.size;
    const existingItem = cart.find(item => item.id === productId && item.size === size);

    if (existingItem) {
        existingItem.quantity += 1;
    } else {
        cart.push({
            id: productId,
            size: size,
            quantity: 1,
            ...product
        });
    }

    saveCart();
    updateCartCount();
    showNotification('Product added to cart!');
    closeProductModal();
}

function toggleWishlistFromModal(productId) {
    toggleWishlist(productId);
    closeProductModal();
}

function closeProductModal() {
    document.getElementById('product-modal').style.display = 'none';
}

// Newsletter Functions
function initializeNewsletter() {
    const form = document.getElementById('newsletter-form');
    if (!form) return;

    form.addEventListener('submit', function (e) {
        e.preventDefault();

        const email = document.getElementById('newsletter-email').value;
        const messageDiv = document.getElementById('newsletter-message');

        if (validateEmail(email)) {
            messageDiv.textContent = 'Thank you for subscribing to our newsletter!';
            messageDiv.className = 'newsletter-message success';
            messageDiv.style.display = 'block';
            form.reset();

            setTimeout(() => {
                messageDiv.style.display = 'none';
            }, 5000);
        } else {
            messageDiv.textContent = 'Please enter a valid email address.';
            messageDiv.className = 'newsletter-message error';
            messageDiv.style.display = 'block';
        }
    });
}

function validateEmail(email) {
    const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return re.test(email);
}





// Notification system
function showNotification(message, type = 'info') {
    // Remove any existing notifications
    const existingNotifications = document.querySelectorAll('.notification');
    existingNotifications.forEach(notification => notification.remove());

    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span class="notification-message">${message}</span>
            <button class="notification-close">&times;</button>
        </div>
    `;

    document.body.appendChild(notification);

    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);

    // Handle close button
    const closeBtn = notification.querySelector('.notification-close');
    closeBtn.addEventListener('click', () => {
        notification.remove();
    });
}

// Event Listeners
function initializeEventListeners() {
    // Cart overlay
    const cartIcon = document.getElementById('cart-icon');
    const cartOverlay = document.getElementById('cart-overlay');
    const closeCart = document.getElementById('close-cart');

    if (cartIcon) {
        cartIcon.addEventListener('click', function (e) {
            e.preventDefault();
            cartOverlay.style.display = 'block';
            renderCartItems();
        });
    }

    if (closeCart) {
        closeCart.addEventListener('click', function () {
            cartOverlay.style.display = 'none';
        });
    }

    // Wishlist overlay
    const wishlistIcon = document.getElementById('wishlist-icon');
    const wishlistOverlay = document.getElementById('wishlist-overlay');
    const closeWishlist = document.getElementById('close-wishlist');

    if (wishlistIcon) {
        wishlistIcon.addEventListener('click', function (e) {
            e.preventDefault();
            wishlistOverlay.style.display = 'block';
            renderWishlistItems();
        });
    }

    if (closeWishlist) {
        closeWishlist.addEventListener('click', function () {
            wishlistOverlay.style.display = 'none';
        });
    }

    // Product modal
    const productModal = document.getElementById('product-modal');
    const closeModal = document.getElementById('close-modal');

    if (closeModal) {
        closeModal.addEventListener('click', closeProductModal);
    }

    if (productModal) {
        productModal.addEventListener('click', function (e) {
            if (e.target === productModal) {
                closeProductModal();
            }
        });
    }

    // Close overlays when clicking outside
    if (cartOverlay) {
        cartOverlay.addEventListener('click', function (e) {
            if (e.target === cartOverlay) {
                cartOverlay.style.display = 'none';
            }
        });
    }

    if (wishlistOverlay) {
        wishlistOverlay.addEventListener('click', function (e) {
            if (e.target === wishlistOverlay) {
                wishlistOverlay.style.display = 'none';
            }
        });
    }

    // Mobile menu toggle
    const hamburger = document.querySelector('.hamburger');
    const navMenu = document.querySelector('.nav-menu');

    if (hamburger && navMenu) {
        hamburger.addEventListener('click', function () {
            navMenu.classList.toggle('active');
            hamburger.classList.toggle('active');
        });
    }
}

// Utility Functions
function showNotification(message, type = 'success') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 100px;
        right: 20px;
        background: ${type === 'success' ? '#22c55e' : '#ef4444'};
        color: white;
        padding: 1rem 2rem;
        border-radius: 5px;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
        z-index: 3000;
        animation: slideIn 0.3s ease;
    `;

    document.body.appendChild(notification);

    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, 3000);
}

// Add CSS animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    
    .nav-menu.active {
        display: flex !important;
        position: absolute;
        top: 100%;
        left: 0;
        width: 100%;
        background: white;
        flex-direction: column;
        padding: 1rem;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
    }
    
    .hamburger.active span:nth-child(1) {
        transform: rotate(-45deg) translate(-5px, 6px);
    }
    
    .hamburger.active span:nth-child(2) {
        opacity: 0;
    }
    
    .hamburger.active span:nth-child(3) {
        transform: rotate(45deg) translate(-5px, -6px);
    }
`;
document.head.appendChild(style);

// Shop page functions (will be used on shop.html)
function loadShopProducts() {
    const container = document.getElementById('shop-products');
    if (!container) return;

    container.innerHTML = products.map(product => `
        <div class="product-card" onclick="openProductModal(${product.id})">
            <div class="product-image">
                <img src="${product.image}" alt="${product.name}" loading="lazy">
            </div>
            <div class="product-info">
                <h3 class="product-title">${product.name}</h3>
                <p class="product-price">$${product.price}</p>
                <div class="product-actions">
                    <button class="btn btn-primary btn-small" onclick="event.stopPropagation(); addToCart(${product.id})">
                        Add to Cart
                    </button>
                    <button class="btn btn-secondary btn-small" onclick="event.stopPropagation(); toggleWishlist(${product.id})">
                        <i class="fas fa-heart"></i>
                    </button>
                </div>
            </div>
        </div>
    `).join('');
}

function filterProducts() {
    const category = document.getElementById('category-filter')?.value;
    const priceRange = document.getElementById('price-filter')?.value;
    const size = document.getElementById('size-filter')?.value;
    const sort = document.getElementById('sort-filter')?.value;

    let filteredProducts = [...products];

    // Filter by category
    if (category && category !== 'all') {
        filteredProducts = filteredProducts.filter(product =>
            product.category.toLowerCase().includes(category.toLowerCase())
        );
    }

    // Filter by price
    if (priceRange && priceRange !== 'all') {
        const [min, max] = priceRange.split('-').map(Number);
        filteredProducts = filteredProducts.filter(product => {
            if (max) {
                return product.price >= min && product.price <= max;
            } else {
                return product.price >= min;
            }
        });
    }

    // Sort products
    if (sort === 'price-low') {
        filteredProducts.sort((a, b) => a.price - b.price);
    } else if (sort === 'price-high') {
        filteredProducts.sort((a, b) => b.price - a.price);
    } else if (sort === 'popularity') {
        // For demo purposes, we'll use a simple popularity based on product ID
        filteredProducts.sort((a, b) => a.id - b.id);
    }

    // Update display
    const container = document.getElementById('shop-products');
    if (container) {
        if (filteredProducts.length === 0) {
            container.innerHTML = '<p style="text-align: center; color: #6b7280; padding: 2rem; grid-column: 1 / -1;">No products match your filters.</p>';
        } else {
            container.innerHTML = filteredProducts.map(product => `
                <div class="product-card" onclick="openProductModal(${product.id})">
                    <div class="product-image">
                        <img src="${product.image}" alt="${product.name}" loading="lazy">
                    </div>
                    <div class="product-info">
                        <h3 class="product-title">${product.name}</h3>
                        <p class="product-price">$${product.price}</p>
                        <div class="product-actions">
                            <button class="btn btn-primary btn-small" onclick="event.stopPropagation(); addToCart(${product.id})">
                                Add to Cart
                            </button>
                            <button class="btn btn-secondary btn-small" onclick="event.stopPropagation(); toggleWishlist(${product.id})">
                                <i class="fas fa-heart"></i>
                            </button>
                        </div>
                    </div>
                </div>
            `).join('');
        }
    }
}

// Checkout functions
function initializeCheckout() {
    const form = document.getElementById('checkout-form');
    if (!form) return;

    form.addEventListener('submit', function (e) {
        e.preventDefault();

        if (validateCheckoutForm()) {
            showOrderConfirmation();
        }
    });

    // Load cart items in checkout
    renderCheckoutItems();

    // Also update cart count
    updateCartCount();
}

function renderCheckoutItems() {
    const container = document.getElementById('checkout-items');
    if (!container) return;

    if (cart.length === 0) {
        container.innerHTML = '<p style="text-align: center; color: #6b7280; padding: 1rem;">Your cart is empty</p>';
        return;
    }

    container.innerHTML = cart.map(item => `
        <div class="order-item">
            <span>${item.name} (Size: ${item.size}) x${item.quantity}</span>
            <span>$${(item.price * item.quantity).toFixed(2)}</span>
        </div>
    `).join('');

    // Update total
    const total = cart.reduce((sum, item) => sum + (item.price * item.quantity), 0);
    const totalElement = document.querySelector('.order-total span:last-child');
    if (totalElement) {
        totalElement.textContent = `$${total.toFixed(2)}`;
    }
}

function validateCheckoutForm() {
    const requiredFields = ['name', 'email', 'address', 'city', 'state', 'zip', 'phone', 'card-name', 'card-number', 'expiry', 'cvv'];
    let isValid = true;

    requiredFields.forEach(fieldName => {
        const field = document.getElementById(fieldName);
        if (field && !field.value.trim()) {
            field.style.borderColor = '#ef4444';
            isValid = false;
        } else if (field) {
            field.style.borderColor = '#d1d5db';
        }
    });

    // Validate email
    const email = document.getElementById('email');
    if (email && !validateEmail(email.value)) {
        email.style.borderColor = '#ef4444';
        isValid = false;
    }

    if (!isValid) {
        showNotification('Please fill in all required fields correctly', 'error');
    }

    return isValid;
}

function showOrderConfirmation() {
    const confirmation = confirm('Are you sure you want to place this order?');
    if (confirmation) {
        // Clear cart
        cart = [];
        saveCart();
        updateCartCount();

        // Show success message
        alert('Order placed successfully! Thank you for your purchase.');

        // Redirect to home page
        window.location.href = 'index.html';
    }
}

// Contact form functions
function initializeContactForm() {
    const form = document.getElementById('contact-form');
    if (!form) return;

    form.addEventListener('submit', function (e) {
        e.preventDefault();

        const name = document.getElementById('contact-name').value;
        const email = document.getElementById('contact-email').value;
        const subject = document.getElementById('contact-subject').value;
        const message = document.getElementById('contact-message').value;

        if (name && email && subject && message && validateEmail(email)) {
            alert('Thank you for your message! We will get back to you soon.');
            form.reset();
        } else {
            showNotification('Please fill in all fields correctly', 'error');
        }
    });
}

// Initialize page-specific functions based on current page
if (document.body.classList.contains('shop-page')) {
    document.addEventListener('DOMContentLoaded', function () {
        loadShopProducts();

        // Add event listeners to filter controls
        const filterControls = ['category-filter', 'price-filter', 'size-filter', 'sort-filter'];
        filterControls.forEach(id => {
            const element = document.getElementById(id);
            if (element) {
                element.addEventListener('change', filterProducts);
            }
        });
    });
}

if (document.body.classList.contains('checkout-page')) {
    document.addEventListener('DOMContentLoaded', initializeCheckout);
}

if (document.body.classList.contains('contact-page')) {
    document.addEventListener('DOMContentLoaded', initializeContactForm);
}
